// $Id: CFileStream.cpp,v 1.10 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CFileStream.hpp"
#include "../Host/CEndian.hpp"
using Exponent::Host::CEndian;
using Exponent::IO::CFileStream;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CFileStream, CCountedObject);

//	===========================================================================
CFileStream::CFileStream() 
		   : m_file(NULL)
		   , m_streamStatus(e_input)
		   , m_streamIsOpen(false)
		   , m_errorState(false)
		   , m_fileIsLittleEndian(false)
{
	EXPONENT_CLASS_CONSTRUCTION(CFileStream);
	NULL_POINTER(m_file);
	m_streamStatus		 = e_input;
	m_streamIsOpen		 = false;
	m_errorState		 = false;
	m_fileIsLittleEndian = false;
}

//	===========================================================================
CFileStream::CFileStream(const CSystemString &filename, const EStreamMode status, const bool fileIsLittleEndian)
		   : m_file(NULL)
		   , m_streamStatus(e_input)
		   , m_streamIsOpen(false)
		   , m_errorState(false)
		   , m_fileIsLittleEndian(false)
{
	EXPONENT_CLASS_CONSTRUCTION(CFileStream);
	NULL_POINTER(m_file);
	m_streamStatus		 = e_input;
	m_streamIsOpen		 = false;
	m_errorState		 = false;
	m_fileIsLittleEndian = false;

	// Try and open the stream
	this->openStream(filename, status, fileIsLittleEndian);
}

//	===========================================================================
CFileStream::~CFileStream()
{
	EXPONENT_CLASS_DESTRUCTION(CFileStream);
	this->closeStream();
}

//	===========================================================================
bool CFileStream::validForOutput()
{
	return (m_streamIsOpen && m_streamStatus == e_output);
}

//	===========================================================================
bool CFileStream::validForInput()
{
	return (m_streamIsOpen && m_streamStatus == e_input);
}

//	===========================================================================
long CFileStream::getStreamPosition()
{
	return ftell(m_file);
}

//	===========================================================================
bool CFileStream::moveToStreamPosition(const long position)
{
	if (m_file && m_streamIsOpen)
	{
		if (fseek(m_file, position, SEEK_SET) == 0)
		{
			return true;
		}
	}
	return false;
}

//	===========================================================================
bool CFileStream::advanceStream(const long amount)
{
	if (m_file && m_streamIsOpen)
	{
		if (fseek(m_file, amount, SEEK_CUR ) == 0)
		{
			return true;
		}
	}
	return false;
}

//	===========================================================================
bool CFileStream::moveToStreamStart()
{
	if (m_file && m_streamIsOpen)
	{
		rewind(m_file);
		return true;
	}
	return false;
}

//	===========================================================================
long CFileStream::getStreamSize()
{
	// If we can find the stream size
	if (m_file && this->validForInput())
	{
		// STore the original position, so we can go back there
		const long position = ftell(m_file);

		// Seend for the end
		if (fseek(m_file, 0, SEEK_END ) != 0)
		{
			m_errorState = true;
			return -1;
		}

		// Store the new position
		const long size = ftell(m_file);

		// Move back to the original position
		if (fseek(m_file, position, SEEK_SET ) != 0)
		{
			m_errorState = true;
			return -1;
		}

		// We found the size! :)
		return size;
	}

	// Faled!
	return -1;
}

//	===========================================================================
void CFileStream::setStreamEndianess(const bool fileIsLittleEndian)
{
	m_fileIsLittleEndian = fileIsLittleEndian;
}

//	===========================================================================
bool CFileStream::openStream(const CSystemString &path, const EStreamMode mode, const bool fileIsLittleEndian)
{
	// Close the old stream
	this->closeStream();

	// Open the file
	const char *access = (mode == e_input) ? "rb" : "w+b";
	m_file = fopen(path.getString(), access);

	// Check that the file opened properly
	m_streamIsOpen	     = (m_file != NULL);
	m_fileIsLittleEndian = fileIsLittleEndian;
	m_streamStatus		 = mode;

	// We are done!
	return m_streamIsOpen;
}

//	===========================================================================
void CFileStream::closeStream()
{
	// Only if we actually need to
	if (m_file && m_streamIsOpen)
	{
		fclose(m_file);
		NULL_POINTER(m_file);
	}

	// Clears errors obviously....
	m_errorState = false;
}

//	===========================================================================
bool CFileStream::writeDataToStream(const void *data, const size_t numberOfBytes)
{
	// If we can
	if (m_file && this->validForOutput())
	{
		// Write the data
		return (fwrite(data, 1, numberOfBytes, m_file) == numberOfBytes);
	}

	// Failed!
	return false;
}

//	===========================================================================
bool CFileStream::readDataFromStream(void *data, const size_t numberOfBytes)
{
	// If we can
	if (m_file && this->validForInput())
	{
		// Write the data
		return (fread(data, 1, numberOfBytes, m_file) == numberOfBytes);
	}

	// Failed!
	return false;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const char &byte)
{
	if (!this->writeDataToStream(&byte, CFILESTREAM_8BIT_SIZE))
	{
		m_errorState = true;
	}
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const unsigned char &byte)
{
	if (!this->writeDataToStream(&byte, CFILESTREAM_8BIT_SIZE))
	{
		m_errorState = true;
	}
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const bool &byte)
{
	char character = byte ? 1 : 0;
	if (!this->writeDataToStream(&character, CFILESTREAM_8BIT_SIZE))
	{
		m_errorState = true;
	}
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const short &byte)
{
	this->write16Bits((void*)&byte);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const unsigned short &byte)
{
	this->write16Bits((void*)&byte);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const long &value)
{
	this->write32Bits((void*)&value);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const unsigned long &value)
{
	this->write32Bits((void*)&value);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const float &value)
{
	this->write32Bits((void*)&value);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator << (const double &value)
{
	this->write64Bits((void*)&value);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (char &byte)
{
	if (!this->readDataFromStream(&byte, CFILESTREAM_8BIT_SIZE))
	{
		m_errorState = true;
	}
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (unsigned char &byte)
{
	if (!this->readDataFromStream(&byte, CFILESTREAM_8BIT_SIZE))
	{
		m_errorState = true;
	}
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (bool &byte)
{
	char character;
	if (!this->readDataFromStream(&character, CFILESTREAM_8BIT_SIZE))
	{
		m_errorState = true;
	}

	// Convert to boolean
	byte = (character == 1);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (short &byte)
{
	this->read16Bits((void*)&byte);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (unsigned short &byte)
{
	this->read16Bits((void*)&byte);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (long &value)
{
	this->read32Bits((void*)&value);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (unsigned long &value)
{
	this->read32Bits((void*)&value);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (float &value)
{
	this->read32Bits((void*)&value);
	return *this;
}

//	===========================================================================
CFileStream &CFileStream::operator >> (double &value)
{
	this->read64Bits((void*)&value);
	return *this;
}

//	===========================================================================
void CFileStream::write16Bits(const void *value)
{
	if (this->swapBits())
	{
		// Convert to unsigned char
		unsigned char *pointer = (unsigned char *)value;

		// The buffer that we are going to write the value in to
		unsigned char buffer[CFILESTREAM_16BIT_SIZE];

		// Swap the byte order
		buffer[0] = pointer[1];
		buffer[1] = pointer[0];

		// Write the data
		if (!this->writeDataToStream(buffer, CFILESTREAM_16BIT_SIZE))
		{
			m_errorState = true;
		}
	}
	else
	{
		// Write the data
		if (!this->writeDataToStream(value, CFILESTREAM_16BIT_SIZE))
		{
			m_errorState = true;
		}
	}
}

//	===========================================================================
void CFileStream::write32Bits(const void *value)
{
	if (this->swapBits())
	{
		// Convert to char
        char *pointer = (char *)value;

		// The buffer that we are going to write the value in to
		char buffer[CFILESTREAM_32BIT_SIZE];

		// Swap the byte order
        buffer[0] = pointer[3];
        buffer[1] = pointer[2];
        buffer[2] = pointer[1];
        buffer[3] = pointer[0];

		// Write the data
		if (!this->writeDataToStream(buffer, CFILESTREAM_32BIT_SIZE))
		{
			m_errorState = true;
		}
	}
	else
	{
		// Write the data
		if (!this->writeDataToStream(value, CFILESTREAM_32BIT_SIZE))
		{
			m_errorState = true;
		}
	}
}

//	===========================================================================
void CFileStream::write64Bits(const void *value)
{
	if (this->swapBits())
	{
		// Convert to char
        char *pointer = (char *)value;

		// The buffer that we are going to write the value in to
		char buffer[CFILESTREAM_64BIT_SIZE];

		// Swap the byte order
        buffer[0] = pointer[7];
        buffer[1] = pointer[6];
        buffer[2] = pointer[5];
        buffer[3] = pointer[4];
        buffer[4] = pointer[3];
        buffer[5] = pointer[2];
        buffer[6] = pointer[1];
        buffer[7] = pointer[0];

		// Write the data
		if (!this->writeDataToStream(buffer, CFILESTREAM_64BIT_SIZE))
		{
			m_errorState = true;
		}
	}
	else
	{
		// Write the data
		if (!this->writeDataToStream(value, CFILESTREAM_64BIT_SIZE))
		{
			m_errorState = true;
		}
	}
}

//	===========================================================================
bool CFileStream::readShortsFromStream(short *data, const unsigned long numberOfElements)
{
	// Read the data
	if (!this->readDataFromStream(data, CFILESTREAM_16BIT_SIZE * numberOfElements))
	{
		m_errorState = true;
		return false;
	}

	if (this->swapBits())
	{
		for (unsigned long i = 0; i < numberOfElements; i++)
		{
			// Pointer to the data
			unsigned char *pointer    = (unsigned char *)&data[i];
			const unsigned char store = pointer[1];		// Store ptr1 [0, 1] s=1
			pointer[1]				  = pointer[0];		// Copy ptr1  [0, 0]
			pointer[0]				  = store;			// Copy old in [1, 0]
		}
	}

	return true;
}

//	===========================================================================
void CFileStream::read16Bits(void *value)
{
	if (this->swapBits())
	{
		// Convert to unsigned char
		unsigned char *pointer = (unsigned char *)value;

		// The buffer that we are going to write the value in to
		unsigned char buffer[CFILESTREAM_16BIT_SIZE];

		// Read the data
		if (!this->readDataFromStream(buffer, CFILESTREAM_16BIT_SIZE))
		{
			m_errorState = true;
		}

		// Swap the byte order
		pointer[0] = buffer[1];
		pointer[1] = buffer[0];
	}
	else
	{
		// Read the data
		if (!this->readDataFromStream(value, CFILESTREAM_16BIT_SIZE))
		{
			m_errorState = true;
		}
	}
}

//	===========================================================================
void CFileStream::read32Bits(void *value)
{
	if (this->swapBits())
    {
		// Convert to unsigned char
		unsigned char *pointer = (unsigned char *)value;

		// The buffer that we are going to write the value in to
		unsigned char buffer[CFILESTREAM_32BIT_SIZE];

		// Read the data
		if (!this->readDataFromStream(buffer, CFILESTREAM_32BIT_SIZE))
		{
			m_errorState = true;
		}

		// swap the byte order
        pointer[0] = buffer[3];
        pointer[1] = buffer[2];
        pointer[2] = buffer[1];
        pointer[3] = buffer[0];
    }
    else
    {
        if (!this->readDataFromStream(value, CFILESTREAM_32BIT_SIZE))
		{
			m_errorState = true;
		}
    }
}

//	===========================================================================
void CFileStream::read64Bits(void *value)
{
	if (this->swapBits())
    {
		// Convert to unsigned char
		unsigned char *pointer = (unsigned char *)value;

		// The buffer that we are going to write the value in to
		unsigned char buffer[CFILESTREAM_64BIT_SIZE];

		// Read the data
		if (!this->readDataFromStream(buffer, CFILESTREAM_64BIT_SIZE))
		{
			m_errorState = true;
		}

		// swap the byte order
        pointer[0] = buffer[7];
        pointer[1] = buffer[6];
        pointer[2] = buffer[5];
        pointer[3] = buffer[4];
        pointer[4] = buffer[3];
        pointer[5] = buffer[2];
        pointer[6] = buffer[1];
        pointer[7] = buffer[0];
    }
    else
    {
        if (!this->readDataFromStream(value, CFILESTREAM_64BIT_SIZE))
		{
			m_errorState = true;
		}
    }
}

//	===========================================================================
bool CFileStream::swapBits() const
{
	return (CEndian::getSystemEndianessRunTime() == CEndian::e_bigEndian) == m_fileIsLittleEndian;
}

//	===========================================================================
bool CFileStream::hasReachEndOfFile() const
{
	return feof(m_file) != 0;
}